#
# Ident512: Implements an identification routine for the groups of order 512.
#
# Implementations
#


#####################################################################################
##
## returns an integer that encodes the content of the provided list
##
InstallGlobalFunction(Pack, function(list)
    local r, i;

    if Length(list) = 0 then
        return 0;
    fi;
    list := Flat(list);
    r := list[1] mod 99661;
    for i in list{[2..Length(list)]} do
        r := (r*10+i) mod 99661;
    od;
    return r;
end);


#####################################################################################
##
##  Computes all conjugacyclasses of the group G. All classes of the same length and
##  the same order of its elements are then saved in a list. All these list are then
##  stored in an additional list and returned.
##
InstallGlobalFunction(BuildCoc, function(G)
    local asList, classes, classtyps, sclasstyps, coc, i, j;

    asList := AttributeValueNotSet(AsSSortedList, G);

    classes := OrbitsDomain(G, asList);
    classtyps := List(classes, x->[Order(x[1]), Length(x)]);
    sclasstyps := Set(classtyps);
    #coc is clusters of conjugacy classes
    coc := List(sclasstyps, x->[ ]);
    for i in [1..Length( sclasstyps)] do
        for j in [1..Length(classes)] do
            if sclasstyps[i] = classtyps[j] then
                Add(coc[i], classes[j]);
            fi;
        od;
    od;

    return coc;
end);

#####################################################################################
##
##  Returns a list of all clusters of groups in a given tree. When initializing the
##  only required argument is tree. The information stored for each cluster include
##  the size of the cluster, the path through the tree, a lists of descriptions for
##  performed coc-tests and the ids of the groups.
##
InstallGlobalFunction(ClusterList, function(arg...)
    local L, tree, opt, cl, LG, LpC, Lgen, newopt, i;
    L:=[];

    tree := arg[1];

    if Length(arg) > 1 then
        opt := arg[2];
        if not IsBound(opt.path) then opt.path := []; fi;
        if not IsBound(opt.descl) then opt.descl := []; fi;
        if not IsBound(opt.pClassAndRank) then opt.pClassAndRank := false; fi;
    else
        opt := rec(path := [], descl := [], pClassAndRank := false);
    fi;

    #If a cluster is found, summarise all gathered information and return as record
    if not IsRecord(tree) then
        cl := rec(size := Size(tree),
                  path := opt.path,
                  descl := opt.descl,
                  ids := tree);#List(tree,t->t.id));

        if opt.pClassAndRank and cl.size > 1 then
            LG := List(tree, id -> SmallGroup(512,id));
            LpC := List(LG, G -> Size(PCentralSeries(G))-1);
            if Size(DuplicateFreeList(LpC)) <> 1 then
                #Print("Warning! Cluster with ids", cl.ids, " does not have uniform pClasses!\n");
                cl.pClass := LpC;
            else
                cl.pClass := LpC[1];
            fi;

            Lgen := List(LG, G -> Size(MinimalGeneratingSet(G)));
            if Size(DuplicateFreeList(Lgen)) <> 1 then
                #Print("Warning! Cluster with ids", cl.ids, " does not have uniform ranks!\n");
                cl.rank := Lgen;
            else
                cl.rank := Lgen[1];
            fi;
        fi;
        return cl;
    else
        #If we still have branches to follow, gather path and desc info and proceed
        for i in [1..Size(tree.next)] do
            newopt := rec(path := ShallowCopy(opt.path),
                          descl := ShallowCopy(opt.descl),
                          pClassAndRank := opt.pClassAndRank);
            Add(newopt.path, tree.fp[i]);
            if IsBound(tree.desc) then
                Add(newopt.descl, tree.desc);
            fi;
            Add(L, ClusterList(tree.next[i], newopt));
        od;
    fi;

    return Flat(L);
end);

#####################################################################################
##
##  Performs a test on the group G of size order. The following tests are implemented:
##
##  -"Parent":  Identifies the parent of G via IdGroup.
##
##  -"Center":  Calculates the rank and abelian invariants of the center of G.
##
##  -"AbelInv": Computes the order of each subgroup of derived series and the abelian
##              invariants of the derived factors of G.
##
##  -"Weights": Calculates the weights of a special pcgs of G.
##
##  -"PCSId":   Identifies the subgroups of the p-central series of G via IdGroup.
##
##  -"ElmOrds": Collects the orders of all elements of G.
##
##  -"CocList": Collects the number of conjugacy classes of a G of same length and
##              same representative order (the order of a representative).
##
##  -"SpSubId": Computes the Frattini subgroup, the upper and lower central series of ##              G and identifies these if possible via IdGroup.
##
##  -"MaxSubId":Computes all maximal subgroups of G and identifies them via IdGroup.
##
##  -"SubId":   Calculates all conjugacyclasses of proper subgoups, stores their size
##              and identifies the isomorphism type of each class via IdGroup.
##
##  -"MaxFactId":Calculates all minimal non-trivial subgroups of G and identifies
##               their quotient with G.
##
##  -"CentQuot":Calculates the central quotients and identifies them via IdGoup.
##
##  -"FactId":  Identifies all proper factors of G.
##
##  -"OutId":   Identifies the groups of outer automorphisms of G, either via IdGroup
##              or via invariants.
##
##  -"SubEqv":  Calculates all conjugacyclasses of proper subgoups, stores their size,
##              identifies the the isomorphism types of each class and determines the
##              isomorphism types and conjugacyclasses in G of its maximal subgroups
##              and minimal supergroups.
##
##  This function accepts a record opt of options with the following content
##  - pack: A boolean that determines if Pack() should be applied to the result
##  The default value is opt.pack := true.
##
InstallGlobalFunction( PerformTestOnGroup, function(G, test, opt...)
    local order, fp, PCS, asList, coc, M, p, C, Q, z, CCS, L, T, N, A, O, i;

    #Process handled options
    if Size(opt) = 0 then
        #If not options are given, create default options
        opt := rec(pack := true);
    elif IsRecord(opt[1]) then
        #If options are given, fill unbound otions with default values
        opt := opt[1];
        if not IsBound(opt.pack) then opt.pack := true; fi;
    else
        Error("Inadmissable option: opt must be empty or a record\n");
    fi;

    order := Order(G);

    if not IsPcGroup(G) then G := Image(IsomorphismPcGroup(G)); fi;

    if test = "Parent" then
        #Identify the parent of G
        PCS := PCentralSeries(G);
        fp :=  IdGroup(FactorGroupNC(G, PCS[Size(PCS)-1]));

    elif test = "Center" then
        fp := [RankPGroup(G), AbelianInvariants(Center(G))];

    elif test = "AbelInv" then
        #Calculate abelian invaraints
        fp := List(DerivedSeriesOfGroup(G),x -> [Size(x), AbelianInvariants(x)]);

    elif test = "Weights" then
        fp :=  LGWeights(SpecialPcgs(Pcgs(G)));

    elif test = "PCSId" then
        #Identify the subgroups of p-lower-central serie
        PCS := PCentralSeries(G);
        fp := List(PCS{[2..Size(PCS)]}, IdGroup);

    elif test = "ElmOrds" then
        #Collect the orders of all elements of G
        asList := AttributeValueNotSet(AsSSortedList, G);
        fp := Collected(List(asList, Order));

    elif test = "CocList" then
        #Collect all length of conjugacy classes and order of their elements
        coc := BuildCoc(G);
        fp := List(coc{[2..Length(coc) ]}, x->[Length(x[1]), Length(x)]);

    elif test = "SpSubId" then
        #Identifies the Frattini and proper subgoups of lower and upper central series
        fp := [IdGroup(FrattiniSubgroup(G))];
        Add(fp,List(Filtered(LowerCentralSeries(G),x->Order(x)<order),IdGroup));
        Add(fp,List(Filtered(UpperCentralSeries(G),x->Order(x)<order),IdGroup));

    elif test = "MaxSubId" then
        #Identify and collect the maximal subgroups
        M := MaximalSubgroups(G);
        fp := [];
        for i in [1..Size(M)] do
            if IdGroupsAvailable(Order(M[i])) then
                Add(fp, IdGroup(M[i]));
            else
                #This is just for using this test in the 'OutId'-test
                Add(fp, [Order(M[i]),0]);
            fi;
        od;
        fp := Collected(fp);

    elif test = "SubId" then
        #Identify conjugacyclasses of subgroups
        CCS := Filtered(ConjugacyClassesSubgroups(G),c->Size(c[1])<order);
        fp := Collected(List(CCS,c->[Size(c), IdGroup(c[1])]));

    elif test = "MaxFactId" then
        #Identify and collect the maximal subgroups
        p := PrimePGroup(G);
        C := Center(G);
        M := List(Filtered(AllSubgroups(C), x -> Size(x)=p), x -> G/x);
        fp := [];
        for i in [1..Size(M)] do
            if IdGroupsAvailable(Order(M[i])) then
                Add(fp, IdGroup(M[i]));
            else
                #This is just for using this test in the 'OutId'-test
                Add(fp, [Order(M[i]),0]);
            fi;
        od;
        fp := Collected(fp);

    elif test = "CentQuot" then
        #Identify and collcet the central quotients of G
        C := Filtered(Center(G),z -> Order(z)>1);
        Q := List(C,z -> FactorGroupNC(G,Group(z)));
        fp := [];
        for i in [1..Size(Q)] do
            if IdGroupsAvailable(Order(Q[i])) then
                Add(fp, IdGroup(Q[i]));
            else
                Add(fp, [Order(Q[i]),0]);
            fi;
        od;
        fp := Collected(fp);

    elif test = "FactId" then
        #Identify all proper factors
        N := Filtered(NormalSubgroups(G),x->Size(x)>1);
        fp := Collected(List(N,n->IdGroup(FactorGroupNC(G,n))));

    elif test = "OutId" then
        #Identify, if possible, outer automorphism group of G
        A := PcGroupAutPGroup(AutomorphismGroupPGroup(G));
        O := A/InnerAutGroupPGroup(A);
        if IdGroupsAvailable(Size(O)) then
            fp := IdGroup(O);
        elif Order(O) = order then
            fp := [Size(O),
            PerformTestOnGroup(O, "ElmOrds", rec(pack:=false)),
            PerformTestOnGroup(O, "CocList", rec(pack:=false)),
            PerformTestOnGroup(O, "MaxSubId",rec(pack:=false)),
            PerformTestOnGroup(O, "CentQuot",rec(pac:=false))];
        else
            fp := [Size(O),
            PerformTestOnGroup(O, "ElmOrds", rec(pack:=false)),
            PerformTestOnGroup(O, "CocList", rec(pack:=false))];
        fi;

    elif test = "LattEqv" then
        #Identify conjugacyclasses of subgroups and their maximalsubgroups and minimal
        #supergroups
        L := LatticeSubgroups(G);
        CCS := ConjugacyClassesSubgroups(L);
        fp := List(CCS,x->0);
        #Identify conjugacyclasses of subgroups
        for i in [1..Size(CCS)] do
            if Size(Representative(CCS[i])) < order then
                fp[i] := [Size(CCS[i]), IdGroup(Representative(CCS[i]))];
            else
                fp[i] := [1, 512];
            fi;
        od;
        M := [];
        M[1] := MaximalSubgroupsLattice(L);
        M[2] := MinimalSupergroupsLattice(L);
        #Identify maximal subgroups and minimal supergroups respectivly
        for i in [1..Size(fp)] do
            T := Collected(List(M[1][i],x->x[1]));
            Add(fp[i], Collected(List(T,t->[fp[t[1]][2], t[2]])));
            T := Collected(List(M[2][i],x->x[1]));
            Add(fp[i], Collected(List(T,t->[fp[t[1]][2], t[2]])));
        od;
        fp := SortedList(fp);

    else
        Error("Please enter a valid test! ", test, " is not recognized.");
    fi;

    if opt.pack then fp := Pack(fp); fi;

    return fp;
end);

#####################################################################################
##
##  Performs a test on the groups of clusters in the list CL. The tests are performed
##  by PerformTestOnGroup. CL could either be a cluster list produced by ClusterList
##  or be a list of lists containing ids. In the former case the output is a list of
##  pairs containg the path of a cluster and the subbrunch resulting from the test. In
##  the latter case only a list of subbranches is returned.
##  This function accepts a record opt of options with the following content
##  - pack: a boolean that is handed to PerformTestOnGroup
##  - ign1cl: a boolean that determines if clusters of size 1 should be ignored
##  - order: an integer giving the order of the groups with ids in CL
##  - verbose: an boolean that determines if a progress should be printed
##  The default value is pack:=true, ign1cl:=true, order:=512, verbose:=true.
##
InstallGlobalFunction(PerformTestOnClusterList, function(CL, test, opt...)
    local TR, cl, fp, id, G, PCS, asList, coc, M, C, Q, z, CCS, N, A, O, branch, i, pos, prog, IdList;

    #Process provided options
    if Size(opt) = 0 then
        #If not options are given, create default options
        opt := rec(pack := true, ign1cl := true, order := 512, verbose := true);
    elif IsRecord(opt[1]) then
        #If options are given, fill unbound otions with default values
        opt := opt[1];
        if not IsBound(opt.pack) then opt.pack := true; fi;
        if not IsBound(opt.ign1cl) then opt.ign1cl := true; fi;
        if not IsBound(opt.order) then opt.order := 512; fi;
        if not IsBound(opt.verbose) then opt.verbose := true; fi;
    else
        Error("Inadmissable option: opt must be empty or a record\n");
    fi;

    #Check if CL is just a list of ids and replace them accordingly
    for i in [1..Size(CL)] do
        if not IsRecord(CL[i]) then
            CL[i] := rec(size := Length(CL[i]), ids := CL[i], path := []);
        fi;
    od;

    TR := [];
    prog := 0;
    IdList := Flat(List(CL,cl->cl.ids));
    for cl in CL do
        if opt.ign1cl and cl.size = 1 then continue; fi;

        branch := rec(fp:=[], next:=[]);

        for id in cl.ids do
            #Print progress if opt.verbose=true
            if opt.verbose and prog <> QuoInt(Position(IdList,id)*10,Size(IdList)) then
                prog := QuoInt(Position(IdList,id)*10,Size(IdList));
                Print(prog, "0% \n");
            fi;

            G := SmallGroup(opt.order, id);
            fp := PerformTestOnGroup(G, test, opt);

            #Find position to insert current group in branch
            pos := Position(branch.fp, fp);
            if pos = fail then
                #If fp is new, create new entries in .fp and .next of branch
                Add(branch.fp, fp);
                Add(branch.next, []);
                pos := Length(branch.next);
            fi;
            #Insert id according to pos of fp in branch
            Add(branch.next[pos], id);
        od;

        #If the current cluster is not part of a search tree (has no path in it),
        #simply return the constructed branch otherwise return pair of path and branch
        if Size(cl.path) = 0 then
            Add(TR, branch);
        else
            Add(TR, [cl.path, branch]);
        fi;
    od;

    #Chatch case of initializing search tree
    if Size(TR) = 1 and IsRecord(TR[1]) then
        TR := TR[1];
    fi;

    #Print results into file with opt.filename if wanted
    if IsBound(opt.filename) and opt.filename <> "" then
        PrintTo(StringFormatted("{}_{}.txt", opt.filename, test), "TR:=", TR, ";");
    fi;

    return TR;
end);

#####################################################################################
##  Extends tree by the test results stored in TR. TR is a list that a path and a
##  branch (a tree of depth one), extending the corresponding cluster.
##
InstallGlobalFunction(AddClusterTestResults, function(tree, TR)
    local tr, branch, fp, pos;

    for tr in TR do
        branch := tree;

        #Follow path to cluster
        for fp in tr[1]{[1..Size(tr[1])-1]} do
            pos := Position(branch.fp, fp);
            if IsBool(pos) then Error("Path not contained in tree!\n"); fi;
            branch := branch.next[pos];
        od;

    #Replace old cluster with extension
    branch.next[Position(branch.fp, tr[1][Size(tr[1])])] := tr[2];
    od;

    return tree;
end);

#####################################################################################
##
##  Builds a search tree for the groups with ids in N or the first N groups, if N is
##  a integer, of size order, using the tests with PerformTestOnClusterList.
##
InstallGlobalFunction(BuildSearchTree, function(N, order, tests)
    local tree, test, CL, TR;

    if IsInt(N) then N:=[[1..N]]; fi;
    if not IsList(N[1]) then N:=[N]; fi;

    tree := PerformTestOnClusterList(N, tests[1], rec(order := order, verbose := false));

    for test in tests{[2..Size(tests)]} do
        CL := ClusterList(tree);
        TR := PerformTestOnClusterList(CL, test, rec(order := order, verbose := false));
        AddClusterTestResults(tree, TR);
    od;

    return tree;
end);

#####################################################################################
##
##  Does a parameter search for desc such that evaluating EvalFpCoc on all clusters
##  of Coc results into at least 2 different values. Therefore using desc as an
##  invaraint in a search tree shrinks down the size of a leaf made of Coc.
##
InstallGlobalFunction(FindParametersForCocTest, function(Coc)
    local Para, N, level, ord, para, lfp;

    Para := [];
    N := Size(Coc[1]);
    level := 1;
    ord := List(Coc[1], x -> Order(x[1][1]));

    #Go through all kinds of parameters seperatly
    while level <= 5  do
        if level = 1 then
            #Check if raising rep. of coc[i] to odd power  lands in original class
            Para := Concatenation(List([3,5,7,9],i-> List([2..N], j -> [1,j,i])));

        elif level = 2 then
            #Check if rep. of coc[i] are roots of elements in coc[j]
            Para := Concatenation(List([2..N], i -> List([2..Minimum(Positions(ord,ord[i]))-1], j -> [2,i,j])));

        elif level = 3 then
            #Check if rep. of coc[i] are powers of elements in coc[j]
            Para := Concatenation(List([2..N], i ->     List([Maximum(Positions(ord,ord[i]))+1..N], j -> [3,i,j])));

        elif level = 4 then
            #Check if rep. is the word [a,b]a^2 made of elements of coc[]
            Para := Concatenation(List([2..N], i -> Concatenation(List([2..N], j -> List([2..N], l -> [4,i,j,l])))));

        elif level = 5 then
            Para := Concatenation(List([2..N], i -> Concatenation(List([2..N], j -> List([2..N], l -> [5,i,j,l])))));
        fi;

        #Test if subset of possible parameters has a 'good' element
        for para in Para do
            #Print(para);
            lfp := List(Coc, coc -> Pack(Collected(EvalFpCoc(coc, para))));
            if Size(DuplicateFreeList(lfp)) > 1 then
                return [lfp, para];
            fi;
        od;
        level := level + 1;
    od;

    #If no parameter yields an improvment raise warning and return fail
    Print("Warning: No suitable parameter found!\n");
    return fail;
end);

#####################################################################################
##
##  De- or encodes the description desc for EvalFPCoc, depending if youe provide
##  an integer (decodes to list) or a list of integers (encodes to int)
##
InstallGlobalFunction(DeEncodeDesc, function(desc)
    local ldesc;

    if IsList(desc) then
        while Length(desc) > 2 do
            desc[1]:= Remove(desc, 1) * 100 + desc[1];
        od;
        desc:= desc[1] * 1000 + desc[2];

    elif IsInt(desc) then
        ldesc := [desc mod 1000];
        desc := QuoInt(desc, 1000);
        while desc > 0 do
            Add(ldesc, desc mod 100);
            desc := QuoInt(desc, 100);
        od;
        desc := Reversed(ldesc);
    fi;
    return desc;
end);

#####################################################################################
##
##  Alters coc according the result fp from EvalFPCoc evaluated with desc.
##
InstallGlobalFunction(AlterCocAccordingToFp, function(coc, desc, fp)
    local sfp, newcls, l, j;

    #Alter coc according to fp
    sfp := Set(fp);
    newcls := List(sfp, l -> []);
    for l in [1..Size(sfp)] do
        for j in [1..Size(fp)] do
            if sfp[l] = fp[j] then
                Add(newcls[l], coc[desc[2]][j]);
            fi;
        od;
    od;
    #Replace cluster desc[2] in coc by subclusters
    coc := Concatenation(coc{[1..desc[2]-1]}, newcls, coc{[desc[2]+1..Size(coc)]});

    return coc;
end);

#####################################################################################
##
##  Build a 1-level serachtree to distinguish the groups of order 512 with id
##  in Ids based on there behavior under EvalFpCoc. Past tests with EvalFpCoc
##  which altered the clusters of conjugacyclasses can be recreated via the
##  list desc. The function returns a record made of fp, next and desc.
##
InstallGlobalFunction(CocSplitRoutine, function(Ids, order, desclist)
    local Coc, i, descl, res, Lfp, tree, pos, desc;

    if Size(Ids) = 1 then return Ids; fi;

    #Build list of cocs according to previous tests documented in desclist
    Coc := List(Ids, id -> BuildCoc(SmallGroup(order, id)));
    for i in [1..Size(Coc)] do
        for desc in desclist do
            desc := DeEncodeDesc(desc);
            Coc[i] := AlterCocAccordingToFp(Coc[i], desc, EvalFpCoc(Coc[i], desc));
        od;
    od;

    #Find 'good' parameters and calculate EvalFpCoc together
    res := FindParametersForCocTest(Coc);

    #If parametersearch fails return list of ids
    if res = fail then return Ids; fi;

    Lfp := res[1]; descl := [res[2]];

    #Transfer results of EvalFpCoc into tree
    tree := rec(fp:=[], next:=[], desc:=[]);

    for i in [1..Size(Ids)] do
        pos := Position(tree.fp, Lfp[i]);
        if IsBool(pos) then
            Add(tree.fp, Lfp[i]);
            tree.next[Position(tree.fp, Lfp[i])] := [Ids[i]];
        else
            Add(tree.next[pos], Ids[i]);
        fi;
    od;

    #Encode description of test
    for desc in descl do
        Add(tree.desc, DeEncodeDesc(desc));
    od;

    for i in [1..Size(tree.next)] do
        tree.next[i] := CocSplitRoutine(tree.next[i], order,
                        Concatenation(desclist, tree.desc));
    od;
    return tree;
end);

#####################################################################################
##
##  Tries to split the clusters of clusters in the list CL via CocSplitRoutine.
##  CL needs to be a cluster list produced by ClusterList. The output is a list of
##  pairs containg the path of a cluster and the subbrunch resulting from the test.
##  This function accepts a record opt of options with the following content
##  - pack: a boolean that is handed to PerformTestOnGroup
##  - order: an integer giving the order of the groups with ids in CL
##  - verbose: an boolean that determines if a progress should be printed
##  The default value is pack:=true, ign1cl:=true, order:=512, verbose:=true.
##
InstallGlobalFunction(PerformCocTestOnClusterList, function(CL, opt...)
    local prog, TR, cl, tree;

    #Process handled options
    if Size(opt) = 0 then
        #If not options are given, create default options
        opt := rec(pack := true, order := 512, verbose := true);
    elif IsRecord(opt[1]) then
        #If options are given, fill unbound otions with default values
        opt := opt[1];
        if not IsBound(opt.pack) then opt.pack := true; fi;
        if not IsBound(opt.order) then opt.order := 512; fi;
        if not IsBound(opt.verbose) then opt.verbose := true; fi;
    else
        Error("Inadmissable option: opt must be empty or a record\n");
    fi;

    prog := 0;
    TR := [];
    for cl in CL do

        #Print progress if opt.verbose=true
        if opt.verbose and prog <> QuoInt(Position(CL,cl)*10,Size(CL)) then
            prog := QuoInt(Position(CL,cl)*10,Size(CL));
            Print(prog, "0% \n");
        fi;

        #Ignore clusters of size one
        if cl.size = 1 then continue; fi;

        #Split a cluster and save resulting tree
        tree := CocSplitRoutine(cl.ids, opt.order, []);

        #If result is a list (of ids), the cluster couldn't be
        #splited and result is omitted
        if IsList(tree) then continue; fi;

        Add(TR, [cl.path, tree]);
    od;

   if IsBound(opt.filename) then
        opt.filename := StringFormatted("CocTest_{}.txt", opt.filename);
        PrintTo(opt.filename, "TR:=", TR, ";");
    else
        return TR;
    fi;
end);

#####################################################################################
##
##  Loads the saved serachtree ID_GROUP_512_TREE
##
Read(Filename(DirectoriesPackageLibrary("ident512",""),"data/ID_GROUP_512_TREE.g"));

#####################################################################################
##
##  Identifies the group G of Order 512.
##  This function accepts a record opt of options with the following content
##  - pack: a boolean that is handed to PerformTestOnGroup
##  - tests: a boolean that determines if clusters of size 1 should be ignored
##  - tree: a serachtree that was build using the same tests
##  - anupq: an integer giving the order of the groups with ids in CL
##  The default value is tests:=["MaxSubId", "CentQuot", "ElmOrds", "CocList"],
##  pack:= true, anupq:=true. tree is unbound and we use ID_GROUP_512_TREE.
##
InstallGlobalFunction(IdGroup512 , function(G, opt...)
    local branch, i, fp, pos, coc, res, GT, id;

    #Process handled options
    if Size(opt) = 0 then
        #If not options are given, create default options
        opt := rec(pack := true,
                   tests := ["MaxSubId", "CentQuot", "ElmOrds", "CocList"],
                   anupq := true);
    elif IsRecord(opt[1]) then
        #If options are given, fill unbound otions with default values
        opt := opt[1];
        if not IsBound(opt.pack) then opt.pack := true; fi;
        if not IsBound(opt.tests) then opt.tests := ["MaxSubId", "CentQuot", "ElmOrds", "CocList"]; fi;
        if not IsBound(opt.anupq) then opt.anupq := true; fi;
    else
        Error("Inadmissable option: opt must be empty or a record\n");
    fi;

    if not IsBound(opt.tree) then
        if not IsBound(ID_GROUP_512_TREE) then
            Print("Loading 'ID_GROUP_512_TREE'\n");
            ID_GROUP_512_TREE := rec(fp:=[],next:=[]);
            Read(Filename(DirectoriesPackageLibrary("ident512",""),"gap/ID_GROUP_512_TREE.g"));
        fi;
        branch := ID_GROUP_512_TREE;
    else
        branch := opt.tree;
    fi;

    if not IsPcGroup(G) then
        G := Image(IsomorphismPcGroup(G));
    fi;

    i := 1;

    #Iterate through tree as long we have not reached a leaf and we can test
    while IsRecord(branch) and (i <= Size(opt.tests) or IsBound(branch.desc)) do
        #Either perform a test on group or do coc-test
        if not IsBound(branch.desc) then
            fp := PerformTestOnGroup(G,opt.tests[i]);
        else
            if not IsBound(coc) then coc := BuildCoc(G); fi;
            fp := EvalFpCoc(coc, DeEncodeDesc(branch.desc));
            coc := AlterCocAccordingToFp(coc, DeEncodeDesc(branch.desc), fp);
            fp := Pack(Collected(fp));
        fi;

        #determine next branch to follow
        pos := Position(branch.fp, fp);

        if IsBool(pos) then
            Error("Group not found in tree.\n");
        fi;

        branch := branch.next[pos];
        i := i + 1;
    od;

    if opt.anupq and IsList(branch) and Size(branch) > 1 then
        #Check if remaining candidates have the same character table
        for id in branch{[1..Size(branch)-1]} do
            GT := SmallGroup(512, id);
            if IsPqIsomorphicPGroup(G, GT) then
                return id;
            fi;
        od;
        return Last(branch);
    fi;

    if Size(branch) = 1 then return branch[1]; fi;
    return branch;
end);


#####################################################################################
################################# Determining Twins #################################
#####################################################################################

#####################################################################################
##
## Takes a list of bins with ids of groups of size order and performs test via
##  PerformTestOnGroup on them. Depending on the outcome a bin is split into smaller
##  bins according to the test.
##
InstallGlobalFunction(SplitBinsByTest, function(Bins, order, test)
    local NewBins, bin, fps, newsubbins, id, G, M, i, fp, pos;

    NewBins := [];

    for bin in Bins do

        fps := [];
        newsubbins := [];

        for id in bin do
            G := SmallGroup(order, id);
            fp := PerformTestOnGroup(G, test, rec(pack := false));

            pos := Position(fps, fp);
            if pos = fail then
                Add(fps, fp);
                Add(newsubbins, []);
                pos := Length(fps);
            fi;
            Add(newsubbins[pos], id);
        od;
        Append(NewBins, newsubbins);
    od;

    return NewBins;
end);


#####################################################################################
##
## Performs all necessary tests on all groups of size order to determine all sets of subgroup- and
## factor-equivalent (SFeq) groups.
##
InstallGlobalFunction(CalculateSiblings, function(order)
    local Bins, NewBins;
    Bins := [[1..NumberSmallGroups(order)]];

    #Determine Bins of subgroup-equivalent groups
    NewBins := SplitBinsByTest(Bins, order, "MaxSubId");
    Bins := Filtered(NewBins, bin -> Size(bin) > 1);
    Print("Test 'MaxSubId' found following bins of size greater one:", Collected(List(Bins,Size)), "\n");

    if Size(Bins) = 0 then return Bins; fi;

    NewBins := SplitBinsByTest(Bins, order, "SpSubId");
    Bins := Filtered(NewBins, bin -> Size(bin) > 1);
    Print("Test 'SpSubId' found following bins of size greater one:", Collected(List(Bins,Size)), "\n");

    if Size(Bins) = 0 then return Bins; fi;

    NewBins := SplitBinsByTest(Bins, order, "SubId");
    Bins := Filtered(NewBins, bin -> Size(bin) > 1);
    Print("Test 'SubId' found following bins of size greater one:", Collected(List(Bins,Size)), "\n");

    if Size(Bins) = 0 then return Bins; fi;

    #Determine bins of factor-equivalent groups
    NewBins := SplitBinsByTest(Bins, order, "MaxFactId");
    Print("Test 'MaxFactId' found following bins of size greater one:", Collected(List(Bins,Size)), "\n");
    Bins := Filtered(NewBins, bin -> Size(bin) > 1);

    if Size(Bins) = 0 then return Bins; fi;

    NewBins := SplitBinsByTest(Bins, order, "FactId");
    Bins := Filtered(NewBins, bin -> Size(bin) > 1);
    Print("Test 'FactId' found following bins of size greater one:", Collected(List(Bins,Size)), "\n");

    return Bins;
end);

#####################################################################################
##
##  Loads and returns a list of ids of subgroup- and factor-equivalent groups of
##  order size if available
##
InstallGlobalFunction(SiblingsIds, function(args...)
    local size, filename, ind;

    size := args[1];

    if size in List([1..6],i->2^i) then
        Print("There are no siblings of order ", size, ".\n");
        return fail;
    elif size in [128, 256, 512] then
        filename := Concatenation("Siblings_", String(size),".g");
        filename := Concatenation("data/TwinsData/",filename);
        Read(Filename(DirectoriesPackageLibrary("ident512",""), filename));

        if Length(args) = 1 then
            ind := [1..Length(BINS_OF_IDS)];
        elif Length(args) = 2 and IsList(args[2]) then
            ind := args[2];
            if not IsSubset([1..Length(BINS_OF_IDS)], ind) then
                Error("Please choose a subset of [1..", Length(BINS_OF_IDS),"] as the list of indicies");
            fi;
        else
            Error("SiblingsIds only expects at most 2 arguments");
        fi;

        return BINS_OF_IDS{ind};
    else
        Print("Siblings of size ", size, " are not documented yet.\n");
        return fail;
    fi;
end);

#####################################################################################
##
##  Loads and returns a list siblings of order size if available
##
InstallGlobalFunction(Siblings, function(args...)
    local Ids;

    if Length(args) = 1 then
        Ids := SiblingsIds(args[1]);
    else
        Ids := SiblingsIds(args[1], args[2]);
    fi;

    if Ids = fail then return fail; fi;

    return List(Ids, ids -> List(ids, id -> SmallGroup(512, id)));
end);


#####################################################################################
##
## Determines the Brauer pairs among the bins of ids of groups of size order listet in Bins.
##
InstallGlobalFunction(DetermineBrauerPairs, function(Bins, order)
    local BPs, bin, BP, CT, i, j, bp;
    BPs := [];


    for bin in Bins do
        if Size(bin) = 1 then continue; fi;

        BP := [[1]];
        CT := List(bin, id -> CharacterTable(SmallGroup(order, id)));

        for i in [2..Size(bin)] do
            for j in [1..Size(BP)] do
                if TransformingPermutationsCharacterTables(CT[i], CT[BP[j][1]]) <> fail then
                    #If CT[i] is equivalent to a known CT[j], add i to the Brauer pair BP[j]
                    Add(BP[j], i);
                    break;
                fi;

                #If CT[i] is not equivalent to known CTs, create new potential pair
                if j = Size(BP) then
                    Add(BP,[i]);
                fi;
            od;
        od;

        #Remove all 'Brauer pairs' of size one
        BP := Filtered(BP,x-> Size(x) > 1);

        for bp in BP do
            #Write ids of found Brauer pairs into list of all found Brauer pairs
            Add(BPs, bin{bp});
        od;
    od;

    return BPs;
end);

#####################################################################################
##
##  Loads and returns a list of ids of subgroup- and factor-equivalent Brauer
##  pairs of groups of order size if available.
##
InstallGlobalFunction(TwinsIds, function(args...)
    local size, filename, ind;

    size := args[1];

    if size in List([1..7],i->2^i) then
        Print("There are no twins of order ", size, ".\n");
        return fail;
    elif size in [256, 512] then
        filename := Concatenation("Twins_", String(size),".g");
        filename := Concatenation("data/TwinsData/",filename);
        Read(Filename(DirectoriesPackageLibrary("ident512",""), filename));

        if Length(args) = 1 then
            ind := [1..Length(BINS_OF_IDS)];
        elif Length(args) = 2 and IsList(args[2]) then
            ind := args[2];
            if not IsSubset([1..Length(BINS_OF_IDS)], ind) then
                Error("Please choose a subset of [1..", Length(BINS_OF_IDS),"] as the list of indicies");
            fi;
        else
            Error("TwinsIds only expects at most 2 arguments");
        fi;

        return BINS_OF_IDS{ind};
    else
        Print("Twins of size ", size, " are not documented yet.\n");
        return fail;
    fi;
end);


#####################################################################################
##
##  Loads and returns a list twins of order size if available
##
InstallGlobalFunction(Twins, function(args...)
    local Ids;

    if Length(args) = 1 then
        Ids := TwinsIds(args[1]);
    else
        Ids := TwinsIds(args[1], args[2]);
    fi;

    if Ids = fail then return fail; fi;

    return List(Ids, ids -> List(ids, id -> SmallGroup(512, id)));
end);

#####################################################################################
##
##  Splits list of ids of groups of size order in Bin, if their outer automorphism groups
##  are isomorphic, via IsPqIsomorphicPGroup from the anupq package
InstallGlobalFunction(SplitExactOutEqv, function(Bin, order)
    local OutEq, G, Aut, Out, OE, i, j, oe;

    OutEq := [];

    G := List(Bin, id -> SmallGroup(order, id));
    Aut := List([1..Size(G)], i -> PcGroupAutPGroup(AutomorphismGroupPGroup(G[i])));
    Out := List([1..Size(Aut)], i -> Aut[i]/InnerAutGroupPGroup(Aut[i]));

    OE := [[1]];

    for i in [2..Size(Bin)] do
        for j in [1..Size(OE)] do
            if IsPqIsomorphicPGroup(Out[i], Out[j]) = true then
                Add(OE[j], i);
                break;
            fi;

            #If CT[i] is not equivalent to known CTs, create new potential pair
            if j = Size(OE) then
                Add(OE,[i]);
            fi;
        od;
    od;

    #Remove all 'Brauer pairs' of size one
    OE := Filtered(OE, x -> Size(x) > 1);

    for oe in OE do
            #Write ids of found Brauer pairs into list of all found Brauer pairs
        Add(OutEq, Bin{oe});
    od;

    return OutEq;
end);

#####################################################################################
##
## Takes a list of bins with ids of groups of size order and evaluates which are outer-
## automorphism equivalent. If exact = true, sets of groups which could not be ruled out to
## be equivalent are testet via SplitExactOutEqv. Otherwise, these are simply included.
##
InstallGlobalFunction(BinsOfOutEqGroups, function(Bins, order, exact)
    local NewBins, bin, fps, newsubbins, potential, id, G, M, i, fp, pos;

    NewBins := [];

    for bin in Bins do

        fps := [];
        newsubbins := [];
        potential := [];

        for id in bin do
            G := SmallGroup(order, id);

            fp := PerformTestOnClusterList(G, "OutId");

            pos := Position(fps, fp);

            if pos = fail then
                Add(fps, fp);
                Add(newsubbins, []);
                pos := Length(fps);
            fi;

            Add(newsubbins[pos], id);

            #Note if two or more groups of bin could not be ruled out
            if Size(fp) = 4 and Size(newsubbins[pos]) > 1 then Add(potential, pos); fi;
        od;

        if exact then
            for i in [1..Size(newsubbins)] do
                if Size(newsubbins[i]) = 1 then continue; fi;
                if i in potential then
                    Append(NewBins, SplitExactOutEqv(newsubbins[i], order));
                else
                    Add(NewBins, newsubbins[i]);
                fi;
            od;
        else
            if Size(potential) > 0  then
                #Warning that groups could not be ruled out.
                Print("The sets ", newsubbins{Set(potential)}, " are not necessarly equivalent, but could not be ruled out.\n");
            fi;
            Append(NewBins, Filtered(newsubbins, b -> Size(b) > 1));
        fi;
    od;

    return NewBins;
end);

#####################################################################################
##
##  Loads and returns a list of ids of twins of order size if available
##
InstallGlobalFunction(LoadSuperTwins, function(size)

    if size in List([1..8],i->2^i) then
        Print("There are no twins of size ", size, ".\n");
        return fail;
    elif size = 512 then
        Read(Filename(DirectoriesPackageLibrary("ident512",""),"data/TwinsData/512_Twins_Ids.g"));
        return BINS_OF_IDS;
    else
        Print("Twins of size ", size, " are not documented yet.\n");
        return fail;
    fi;

end);

#####################################################################################
##
##  Creates a list of records that can be exported into latex code
##
InstallGlobalFunction(TableOfGroupInvariants, function(Bins, order)
    local rows, ids, newrow, G, A, O, OutEqv;

    LoadPackage("anupq");

    rows := [];

    for ids in SortedList(Bins) do

        newrow := rec(ids := ids);
        G := List(ids, id -> SmallGroup(order, id));

        #Check if all groups in the bin have same rank
        newrow.rank := DuplicateFreeList(List([1..Size(G)], i -> Rank(G[i])));
        if Length(newrow.rank) > 1 then Error("Not matching ranks"); fi;
        newrow.rank := newrow.rank[1];

        #Check if all groups in the bin have same p-class
        newrow.pclass := DuplicateFreeList(List([1..Size(G)], i -> Length(PCentralSeries(G[i]))-1));
        if Length(newrow.pclass) > 1 then Error("Not matching p-classes"); fi;
        newrow.pclass := newrow.pclass[1];

        A := List([1..Size(G)], i -> PcGroupAutPGroup(AutomorphismGroupPGroup(G[i])));

        #Check if all groups in the bin have equal sized automorphism groups
        newrow.SizeAut := DuplicateFreeList(List([1..Size(G)], i -> Size(A[i])));
        if Length(newrow.SizeAut) > 1 then Error("Not matching sizes of Aut"); fi;
        newrow.SizeAut := newrow.SizeAut[1];

        #Check groups of outer automorphisms
        OutEqv := List([1..Size(G)], i -> PerformTestOnGroup(G[i], "OutId", rec(pack:=false)));

        if Length(DuplicateFreeList(OutEqv)) > 1 then
            #If groups yield different fingerprint, they are not equivalent
            newrow.OutEqv := "no";
        elif Length(OutEqv[1]) = 2 then
            #If they yield same fingerprint and that used IdGroup, they are equivalent
            newrow.OutEqv := "yes";
        else
            if Size(newrow.ids) = 2 then
                #If onyl check a pair of groups check them with anupq. That could take a while.
                O := List([1..Size(G)], i -> A[i]/InnerAutGroupPGroup(A[i]));
                if IsPqIsomorphicPGroup(O[1],O[2]) then
                    newrow.OutEqv := "yes";
                else
                    newrow.OutEqv := "no";
                fi;
            else
                newrow.OutEqv := fail;
            fi;
        fi;

        #Since isoclinism and mip take a while, exclude them here.
        newrow.isoclinic := "";
        newrow.mip := "";

        Add(rows, newrow);
    od;

    return rows;

end);


InstallGlobalFunction(DetermineMIP, function(rows, prime, power)
    local row, tt;
    LoadPackage("modisom");

    for row in rows do
        if Size(row.ids) = 1 then continue; fi;
        tt := MIPSplitGroupsByAlgebras(prime, power, row.ids);
        if Length(tt.bins) = 0 then
            row.mip := "no";
        else
            row.mip := "yes";
        fi;
        AppendTo("MIP_temp.txt", tt, "\n");
    od;
    PrintTo("MIP_rows.txt", rows);
    return rows;
end);


InstallGlobalFunction(DetermineIsoclinic, function(rows, order, exclude)
    local row, pos, G, isocl;
    LoadPackage("xmod");

    for row in rows do
        if Size(row.ids) = 1 then continue; fi;

        pos := Position(List(exclude, l -> l[1]), row.ids);
        if pos <> fail then
            row.isoclinic := exclude[pos][2];
            continue;
        fi;

        G := List(row.ids, id -> SmallGroup(order, id));

        isocl := Isoclinism(G[1], G[2]);

        if isocl = fail then
            row.isoclinic := "no";
        else
            row.isoclinic := "yes";
        fi;

        AppendTo("IP_temp.txt", [row.ids, isocl]);
    od;

    PrintTo("Ip_rows.txt", rows, ";");

    return rows;
end);


InstallGlobalFunction(MergeTablesOfInvariants, function(tab1, tab2)
    local tab, row, pos, new, name;

    tab := [];

    if Size(tab1) <> Size(tab2) then Error("Both provided tables must have the same length"); fi;

    for row in tab1 do
        pos := Position(List(tab2, t -> t.ids), row.ids);

        if pos = fail then
            Error(Position(tab1, row), "th row of tab1 has no correspondance in tab2");
        fi;

        new := rec(ids := [], rank := 0, pclass:=0, isoclinic:="", SizeAut:=0, OutEqv:="", MIP:="");

        #Check for all entries of new, if tab1 and tab2 are equal or one of them is on default
        for name in RecNames(new) do
            if \.(row, RNamObj(name)) = \.(tab2[pos], RNamObj(name)) then
                \.\:\=( new, RNamObj(name), \.(row, RNamObj(name)));
            elif \.(row, RNamObj(name)) in [[],0,""] then
                \.\:\=(new, RNamObj(name), \.(tab2[pos], RNamObj(name)));
            elif \.(tab2[pos], RNamObj(name)) in [[],0,""] then
                \.\:\=(new, RNamObj(name), \.(row, RNamObj(name)));
            else
                Error("I am not sure what happend");
            fi;
        od;

        Add(tab, new);
    od;

    return tab;
end);


InstallGlobalFunction(PrintTableToFile, function(rows, filename)
    local head, row, i;
    filename := Concatenation(filename, "_table.txt");
    head := "Pair & Rank & $p$-class& Isoclinic & Size of Aut & Out-equiv & MIP \\\\ \n";

    PrintTo(filename, head);

    for row in rows do

        AppendTo(filename, "(",row.ids[1]);
        for i in [2..Size(row.ids)] do AppendTo(filename, ", ", row.ids[i]); od;
        AppendTo(filename, ") & ");

        AppendTo(filename, row.rank, " & ",
                           row.pclass, " & ",
                           row.isoclinic, " & ",
                           row.SizeAut, " & ",
                           row.OutEqv, " & ",
                           row.mip,
                           " \\\\ \n");
    od;
end);
