#
# Ident512: Implements an identification routine for the groups of order 512.
#
#! @Chapter Introduction
#!
#! The main purpose of the &Ident512; package is to implement an identification routine
#! for the groups of order 512. This is done by calculating a number of invariants and comparing
#! the results with the values of all possible candidates via a precalculated search tree.
#! This Procedure determins the vast majority of groups.
#! In the unfavorable case that the used invariants shrink the number of candidates down
#! to more than one group the function `IsPqIsomorphicPGroup` from the &Anupq; package is
#! used to determine the correct candidate.
#!
#! All functions used to build the precalculated search tree are also content of this package.
#! Thus one can build ones own search tree with a different set of invariants.
#!
#! While determining suitable invariants for this task, the definition of subgroup- and
#! factor-equivlance was developed together with the definition of twins (see ). The latter
#! part of the packag implements function to determine these sets of groups, load precalculated
#! lists of such groups and calculate certain invariants for these sets.
#!
#! @Chapter Functionality
#! This chapter lists all functions contained in the &Ident512; package.
#!
#! @Section Functions for building a search tree
#!
#! This section describes all functions and variables connected to building a
#! search tree to identify groups via implemented invariants.

#! @Arguments list
#! @Returns An interger encoding the content of <A>list</A>.
#! @Description
#! Encodes the content of the provided <A>list</A> into an intger.
#! @BeginExample
#!Pack([1,2,3]);
#! 123
#!Pack([606, 109, 1405]);
#! 63095
#!Pack([[1,2,3], [606, 1405]]);
#! 19765
#! @EndExample
DeclareGlobalFunction("Pack");

#! @Arguments G
#! @Returns A list of lists containing the conjugacy classes of <A>G</A>.
#! @Description
#! Computes the conjugacy classes of the group <A>G</A> and clusters them such that each clusters
#! contains all classes of a certain length and certain representative order (meaning the order
#! of any represantative).
DeclareGlobalFunction("BuildCoc");

#! @Arguments tree [,...]
#! @Returns A list of all clusters of groups in a given serach tree (the leafs of the tree).
#! @Description
#! When initializing the only required argument is <A>tree</A>. The information stored for each
#! cluster includs the size of the cluster, the path through the tree, a lists of descriptions
#! for performed `EvalFpCoc`-tests and the ids of the groups.
#! @BeginExample
#!tree := rec( fp := [ 63049, 75721, 4310 ], next := [ rec( fp := [ 337 ], next := [ [ 3, 8 ] ] ),
#!                [ 4 ], rec( fp := [ 32173 ], next := [ [ 5, 6, 7 ] ] ) ] );;
#!ClusterList(tree);
#! [ rec( descl := [  ], ids := [ 3, 8 ], path := [ 63049, 337 ], size := 2 ),
#!   rec( descl := [  ], ids := [ 4 ], path := [ 75721 ], size := 1 ),
#!   rec( descl := [  ], ids := [ 5, 6, 7 ], path := [ 4310, 32173 ], size := 3 ) ]
#! @EndExample
DeclareGlobalFunction("ClusterList");

#! @Arguments G, test [,opt]
#! @Returns The value returned by <A>test</A>, if desried encoded as an integer.
#! @Description
#! Performs <A>test</A> on the group <A>G</A>. The following tests are implemented:
#! * <A>"Parent"</A>: Identifies the parent of <A>G</A> via `IdGroup`.
#! * <A>"Center"</A>: Calculates the rank and abelian invariants of the center of <A>G</A>.
#! * <A>"AbelInv"</A>: Computes the order of each term of the derived series and the abelian
#!                      invariants of the derived factors of <A>G</A>.
#! * <A>"Weights"</A>: Calculates the `LGWeights` of a special pcgs of <A>G</A>.
#! * <A>"PCSId"</A>: Identifies the proper subgroups of the p-central series of <A>G</A>
#!                   via `IdGroup`.
#! * <A>"ElmOrds"</A>: Collects the orders of all elements of <A>G</A>.
#! * <A>"CocList"</A>: Collects the number of conjugacy classes of a <A>G</A> of same length and
#!                     same representative order (the order of a representative).
#! * <A>"SpSubId"</A>: Computes the Frattini subgroup, the upper and lower central series of
#!                     <A>G</A> and identifies all terms if possible via `IdGroup`.
#! * <A>"MaxSubId"</A>: Computes all maximal subgroups of <A>G</A> and identifies them via `IdGroup`.
#! * <A>"SubId"</A>: Collects all sizes and isormorphism types of conjugacyclasses of proper
#!                   subgoups via `IdGroup`.
#! * <A>"MaxFactId"</A>: Calculates all minimal non-trivial subgroups of G and identifies their
#!                       quotient with <A>G</A> via `IdGroup`.
#! * <A>"CentQuot"</A>: Calculates the central quotients of <A>G</A> and identifies them
#!                      via `IdGroup`.
#! * <A>"FactId"</A>: Identifies all proper factors of <A>G</A> via `IdGroup`.
#! * <A>"OutId"</A>: Tries to identifies the groups of outer automorphisms of <A>G</A>, either via
#!                   `IdGroup` or via invariants.
#! * <A>"LattEqv"</A>: Calculates all conjugacy classes of proper subgoups of <A>G</A>, stores
#!                     their size, identifies the the isomorphism types of each class and
#!                     determines the isomorphism types and conjugacy classes in <A>G</A> of its
#!                     maximal subgroups and minimal supergroups.
#! This function accepts a record <A>opt</A> of options with the single content <A>opt.pack</A>,
#! a boolean that determines if <C>Pack</C> should be applied to the result of a test. The
#! default value of pack is <K>true</K>.
#! @BeginExample
#!G := SmallGroup(512, 1405);;
#!PerformTestOnGroup(G, "ElmOrds");
#! 85416
#!PerformTestOnGroup(G, "ElmOrds", rec(pack := false));
#! [ [ 1, 1 ], [ 2, 7 ], [ 4, 24 ], [ 8, 288 ], [ 16, 64 ], [ 32, 128 ] ]
#! @EndExample
DeclareGlobalFunction("PerformTestOnGroup");

#! @Arguments CL, test [,opt]
#! @Returns Either a list of pairs each containg the path of the respective cluster and the
#!subbrunch resulting from <A>test</A> or just a list of subbranches.
#! @Description
#! Performs a test on the groups of clusters in the list <A>CL</A>. The tests are performed
#! by `PerformTestOnGroup`. <A>CL</A> could either be a cluster list produced by
#! `ClusterList` or be a list of lists containing ids. In the former case the output is
#! a list of pairs containg the path of a cluster and the subbrunch resulting from the test.
#! In the latter case only a list of subbranches is returned.
#! This function accepts a record <A>opt</A> of options with the following content:
#! * <A>pack</A>: A boolean that is handed to `PerformTestOnGroup`.
#! * <A>ign1cl</A>: A boolean that determines if clusters of size 1 should be ignored.
#! * <A>order</A>: An integer giving the order of the groups with ids in <A>CL</A>.
#! * <A>verbose</A>: An boolean that determines if a progress should be printed.
#! The default value is <A>pack</A>:=<K>true</K>, <A>ign1cl</A>:=<K>true</K>, <A>order</A>:=512,
#! <A>verbose</A>:=<K>true</K>.
#! @BeginExample
#!CL:=[[1..5],[6..10]];;
#!PerformTestOnClusterList(CL,"ElmOrds",rec(verbose:=false));
#! [ rec( fp := [ 88570, 4310, 63049, 75721 ],next := [ [ 1 ], [ 2, 5 ], [ 3 ], [ 4 ] ] ),
#!   rec( fp := [ 4310, 63049 ], next := [ [ 6, 7, 9, 10 ], [ 8 ] ] ) ]
#! @EndExample
DeclareGlobalFunction("PerformTestOnClusterList");

#! @Arguments tree, TR
#! @Returns A search tree (implemented as a record).
#! @Description
#! Extends <A>tree</A> by the test results stored in <A>TR</A>, a list of pairs each containing
#! the path in <A>tree</A> and a branch (a tree of depth one), extending the corresponding cluster.
#! @BeginExample
#!tree:=BuildSearchTree([3,4,5,6,7,8], 512, ["ElmOrds"]);;
#!CL := ClusterList(tree);
#! [ rec( descl := [  ], ids := [ 3, 8 ], path := [ 63049 ], size := 2 ),
#!   rec( descl := [  ], ids := [ 4 ], path := [ 75721 ], size := 1 ),
#!   rec( descl := [  ], ids := [ 5, 6, 7 ], path := [ 4310 ], size := 3 ) ]
#!TR:=PerformTestOnClusterList(CL,"MaxSubId",rec(verbose:=false));
#! [ [ [ 63049 ], rec( fp := [ 62603 ], next := [ [ 3, 8 ] ] ) ],
#!   [ [ 4310 ], rec( fp := [ 11744, 86646 ], next := [ [ 5 ], [ 6, 7 ] ] ) ] ]
#!AddClusterTestResults(tree,TR);;
#!ClusterList(tree);
#! [ rec( descl := [  ], ids := [ 3, 8 ], path := [ 63049, 62603 ], size := 2 ),
#!   rec( descl := [  ], ids := [ 4 ], path := [ 75721 ], size := 1 ),
#!   rec( descl := [  ], ids := [ 5 ], path := [ 4310, 11744 ], size := 1 ),
#!   rec( descl := [  ], ids := [ 6, 7 ], path := [ 4310, 86646 ], size := 2 ) ]
#! @EndExample
DeclareGlobalFunction("AddClusterTestResults");

#! @Arguments N, order, tests
#! @Returns A search tree (implemented as a record).
#! @Description
#! Builds a search tree for the groups with ids in <A>N</A> or, if <A>N</A> is an integer,
#! the first <A>N</A> groups of size <A>order</A>, using the tests from the list <A>tests</A> and
#! `PerformTestOnClusterList`.
#! @BeginExample
#!BuildSearchTree(10, 512, ["MaxSubId", "CentQuot"]);
#! rec( fp := [ 25611, 76646, 62603, 89031, 11744, 86646, 63803, 78015 ],
#!     next := [ [ 1 ], [ 2 ], rec( fp := [ 63542, 75796 ], next := [ [ 3 ], [ 8 ] ] ), [ 4 ], [ 5 ],
#!      rec( fp := [ 30861, 61235 ], next := [ [ 6 ], [ 7 ] ] ), [ 9 ], [ 10 ] ] )
#!BuildSearchTree([1,2,3,5,7,11], 512, ["ElmOrds", "CocList"]);
#! rec( fp := [ 88570, 4310, 63049, 94635 ],
#!     next := [ [ 1 ], rec( fp := [ 96561, 32173 ], next := [ [ 2 ], [ 5, 7 ] ] ), [ 3 ], [ 11 ] ] )
#! @EndExample
DeclareGlobalFunction("BuildSearchTree");

#! @Arguments coc
#! @Returns A pair made of a list of return values from `EvalFpCoc` and a list describing the test
#! performed by `EvalFpCoc`.
#! @Description
#! Does a parameter search such that evaluating `EvalFpCoc` on the clusters of conjugacy classes
#! in <A>coc</A> results into at least 2 different values.
#! @BeginExample
#!coc:=List([3, 8],id->BuildCoc(SmallGroup(512, id)));;
#!FindParametersForCocTest(coc);
#! [ [ 1363, 46760 ], [ 3, 2, 4 ] ]
#! @EndExample
DeclareGlobalFunction("FindParametersForCocTest");

#! @Arguments desc
#! @Returns An interger or a list of integers
#! @Description
#! De- or encodes the description for `EvalFPCoc`, depending if
#! an integer (decodes to a list) or a list of integers (encodes to an integer) is provided.
#! @BeginExample
#!DeEncodeDesc([3,2,4]);
#! 302004
#!DeEncodeDesc(302004);
#! [3, 2, 4]
#! @EndExample
DeclareGlobalFunction("DeEncodeDesc");

#! @Arguments coc, desc, fp
#! @Returns A list of lists containing the conjugacy classes
#! @Description
#! Alters <A>coc</A> according the result <A>fp</A> from <C>EvalFPCoc</C> evaluated with
#! <A>desc</A>.
DeclareGlobalFunction("AlterCocAccordingToFp");

#! @Arguments Ids, order, desclist
#! @Returns A serach tree (implemented as a record) of depth one.
#! @Description
#! Builds a search tree of depth one to distinguish the groups of <A>order</A> with ids
#! in <A>Ids</A> based on the returned values of `EvalFpCoc`. Past tests with
#! `EvalFpCoc` which altered the clusters of conjugacy classes returned by `BuildCoc` can be
#! recreated via the list <A>desclist</A>.
#! @BeginExample
#!CocSplitRoutine([5,6,7], 512, []);
#! rec( desc := [104003], fp := [ 1214, 16 ],
#!      next := [ [ 5 ], rec( desc := [40204005], fp := [ 20792, 46881 ], next := [ [ 6 ], [ 7 ] ] ) ] )
#! @EndExample
DeclareGlobalFunction("CocSplitRoutine");

#! @Arguments CL [,opt]
#! @Returns A list of pairs each containg the path of the respective cluster and the subbrunch
#!          resulting from `CocSplitRoutine`.
#! @Description
#! Tries to split the clusters in the list <A>CL</A> via `CocSplitRoutine`.
#! <A>CL</A> needs to be a cluster list produced by `ClusterList`. This function accepts a record
#! <A>opt</A> of options with the following content:
#! * <A>pack</A>: A boolean that determines if <C>Pack</C> should be applied to the result.
#! * <A>order</A>: An integer giving the order of the groups with ids in <A>CL</A>.
#! * <A>verbose</A>: An boolean that determines if a progress should be printed.
#! The default value is <A>pack</A>:=<K>true</K>, <A>order</A>:=512,
#! <A>verbose</A>:=<K>true</K>.
#! @BeginExample
#!tree:=PerformTestOnClusterList([[3,8,5,6,7]],"CocList",rec(verbose:=false));;
#!CL:=ClusterList(tree);
#! [ rec( descl := [  ], ids := [ 3, 8 ], path := [ 337 ], size := 2 ),
#!   rec( descl := [  ], ids := [ 5, 6, 7 ], path := [ 32173 ], size := 3 ) ]
#!TR:=PerformCocTestOnClusterList(CL,rec(verbose:=false));;
#!AddClusterTestResults(tree, TR);;
#!ClusterList(tree);
#! [ rec( descl := [ [ 302004 ] ], ids := [ 3 ], path := [ 337, 1363 ], size := 1 ),
#!   rec( descl := [ [ 302004 ] ], ids := [ 8 ], path := [ 337, 46760 ], size := 1 ),
#!   rec( descl := [ [ 104003 ] ], ids := [ 5 ], path := [ 32173, 1214 ], size := 1 ),
#!   rec( descl := [ [ 104003 ], [ 40204005 ] ], ids := [ 6 ], path := [ 32173, 16, 20792 ], size := 1 ),
#!   rec( descl := [ [ 104003 ], [ 40204005 ] ], ids := [ 7 ], path := [ 32173, 16, 46881 ], size := 1 ) ]
#! @EndExample
DeclareGlobalFunction("PerformCocTestOnClusterList");

#! @Section Identifying groups via a serach tree
#!
#! This section briefly describes the function and variable in this package used to
#! identify a arbitrary group of order 512.

#! @Description
#! The serach tree of groups of order 512 that can determine the vast majority of groups via the
#! packed invariants <A>"MaxSubId"</A>, <A>"CentQuot"</A>, <A>"ElmOrds"</A>, <A>"CocList"</A> from
#! `PerformTestOnGroup` and the results from `CocSplitRoutine`.
DeclareGlobalName("ID_GROUP_512_TREE");

#! @Arguments G [,opt]
#! @Returns The id of the group of order 512 in `SmallGroup` isomorphic to <A>G</A>.
#! @Description
#! Identifies the group <A>G</A> of order 512.
#! This function accepts a record <A>opt</A> of options with the following content:
#! * <A>pack</A>: A boolean that is handed to `PerformTestOnGroup`.
#! * <A>tests</A>: A list of strings, that determines which tests from `PerformTestOnGroup` are
#!                 used to distinguish groups with <A>tree</A>.
#! * <A>tree</A>: A serachtree that was build using the invariants resulting from <A>tests</A>.
#! * <A>anupq</A>: A boolean that determines if the <A>IsPqIsomorphicPGroup</A> from the
#!                 &Anupq;-Package should be used to distinguish remaining candidates for the
#! identity of <A>G</A> if necessary.
#! The default value is <A>tests</A>:=[<A>"MaxSubId"</A>, <A>"CentQuot"</A>, <A>"ElmOrds"</A>, <A>"CocList"</A>],
#! <A>pack</A>:= <K>true</K>, <A>anupq</A>:=<K>true</K>. tree is unbound and
#! `ID_GROUP_512_TREE` is used by default.
#! @BeginExample
#!G := Image(IsomorphismPermGroup(SmallGroup(512, 1984)));;
#!IdGroup512(G);
#! 1984
#!H := Image(IsomorphismPermGroup(SmallGroup(512, 449)));;
#!IdGroup512(H, rec(anupq := false));
#! [449, 450]
#!IdGroup512(H, rec(anupq := true));
#! 449
#! @EndExample
DeclareGlobalFunction("IdGroup512");

#! @Section Functions to determine siblings and twins
#!
#! This section describes all functions to calculate
#! * subgroup-equivalent groups, i.e. groups with isomorphic proper subgroups,
#! * factor-equivalent groups,  i.e. groups with isomorphic proper factors,
#! * siblings, i.e. groups that are subgroup- and factor equivalent and where the bijection of
#!   subgroups preservs conjugacy and maps the Frattini subgroups, lower and upper central series
#!   onto the respective counterparts,
#! * twins, siblings that are also Brauer Pairs.
#! It also provides functions to load precalculated
#! lists of such groups and functions to calculate invariants of these sets of groups and a way
#! to export latex-tables-rows with these informations

#! @Arguments Bins, order, test
#! @Returns A list of sublists of the provided lists in <A>Bins</A>.
#! @Description
#! Takes a list of bins with ids of groups of size <A>order</A> and performs <A>test</A> via
#! `PerformTestOnGroup` on them. Depending on the outcome a bin is split into smaller
#! bins according to the test result.
#! @BeginExample
#!SplitBinsByTest([[1..10]], 512, "MaxSubId");
#! [ [ 1 ], [ 2 ], [ 3, 8 ], [ 4 ], [ 5 ], [ 6, 7 ], [ 9 ], [ 10 ] ]
#! @EndExample
DeclareGlobalFunction("SplitBinsByTest");

#! @Arguments order
#! @Returns A list of lists of ids of groups of size <A>order</A> that are siblings.
#! @Description
#! Performs all necessary tests on all groups of size <A>order</A> to determine all sets of
#! Siblings.
DeclareGlobalFunction("CalculateSiblings");

DeclareGlobalName("BINS_OF_IDS");

#! @Arguments order [,indices]
#! @Returns A list of lists of ids of groups of size <A>order</A> that are siblings.
#! @Description
#! Loads and returns a list of ids of siblings of size <A>order</A> if available. If a
#! list <A>indices</A> is given only the respective entries are returned.
#! @BeginExample
#!SiblingsIds(128);
#! [ [ 1317, 1322 ], [ 1327, 1329 ], [ 1597, 1598 ] ]
#!SiblingsIds(256, [1..3]);
#! [ [ 227, 228 ], [ 258, 260, 261 ], [ 845, 846 ] ]
#! @EndExample
DeclareGlobalFunction("SiblingsIds");

#! @Arguments order [,indices]
#! @Returns A list of groups of size <A>order</A> that are siblings.
#! @Description
#! Loads and returns a list of ids of siblings of size <A>order</A> if available. If
#! <A>indices</A> are given only the respective entries are returned.
DeclareGlobalFunction("Siblings");

#! @Arguments Bins, order
#! @Returns A list of sublists of the provided lists in <A>Bins</A>.
#! @Description
#! Determines the Brauer pairs among the bins of ids of groups of size <A>order</A> listet
#! in <A>Bins</A>.
DeclareGlobalFunction("DetermineBrauerPairs");

#! @Arguments order [,indices]
#! @Returns A list of lists of ids of groups of size <A>order</A> that are twins.
#! @Description
#! Loads and returns a list of ids of twins of size <A>order</A> if available. If a
#! list <A>indices</A> is given only the respective entries are returned.
#! @BeginExample
#!TwinsIds(256);
#! [ [ 1734, 1735 ], [ 1736, 1737 ], [ 1739, 1740 ], [ 1741, 1742 ], [ 3678, 3679 ], [ 4154, 4157 ], [ 4155, 4158 ], [ 4156, 4159 ] ]
#!TwinsIds(256, [1..3]);
#! [ [ 1734, 1735 ], [ 1736, 1737 ], [ 1739, 1740 ] ]
#! @EndExample
DeclareGlobalFunction("TwinsIds");

#! @Arguments order [,indices]
#! @Returns A list of groups of size <A>order</A> that are twins.
#! @Description
#! Loads and returns a list of twins of size <A>order</A> if available. If a
#! list <A>indices</A> is given only the respective entries are returned.
DeclareGlobalFunction("Twins");

#! @Arguments Bin, order
#! @Returns Sublists of the provided <A>Bin</A> containing the outer-automorphism equivalent groups
#! among them.
#! @Description
#! Splits list of ids of groups of size <A>order</A> in <A>Bin</A>, if their outer automorphism groups
#!  are isomorphic, via `IsPqIsomorphicPGroup` from the &Anupq; package.
DeclareGlobalFunction("SplitExactOutEqv");

#! @Arguments Bins, order, exact
#! @Returns A list of sublists of the provided lists in <A>Bins</A>.
#! @Description
#! Takes a list of bins with ids of groups of size <A>order</A> and evaluates which are outer-
#! automorphism equivalent. If <A>exact</A> = <K>true</K>, sets of groups which could not be ruled out
#! to be equivalent are testet via `SplitExactOutEqv`. Otherwise these sets are simply included.
DeclareGlobalFunction("BinsOfOutEqGroups");

#! @Arguments order
#! @Returns A list of lists of ids of twins of size <A>order</A> that are outer-
#! automorphism equivalent.
#! @Description
#! Loads and returns a list of ids of twins of size <A>order</A> if available.
#! @BeginExample
#!LoadSuperTwins(512);
#! [ [ 28340, 28341 ], [ 1364, 1368 ], [ 1365, 1369 ], [ 1366, 1370 ], [ 1367, 1371 ], [ 1682, 1683 ],
#!  [ 1680, 1681 ], [ 1746, 1747 ], [ 1748, 1749 ], [ 1750, 1751 ], [ 1752, 1753 ], [ 53283, 53285 ],
#!  [ 43261, 43262 ], [ 28426, 28427 ], [ 29033, 29034 ], [ 29035, 29036 ], [ 30917, 30923 ],
#!  [ 30981, 30983 ], [ 32666, 32667 ], [ 53282, 53284 ], [ 139350, 139355 ], [ 53396, 53398 ],
#!  [ 53382, 53384 ], [ 42518, 42519 ], [ 42482, 42483 ], [ 42520, 42521 ], [ 43208, 43209 ],
#!  [ 45383, 45384 ], [ 53306, 53307 ], [ 53383, 53385 ], [ 53397, 53399 ], [ 53484, 53485 ],
#!  [ 253429, 253432 ], [ 253447, 253450 ] ]
#! @EndExample
DeclareGlobalFunction("LoadSuperTwins");

#! @Arguments bins, order
#! @Returns A list of records
#! @Description
#! Calculates a number of invariants and characteristics for the groups of order <A>order</A> with
#! ids in lists of <A>bins</A>. If the invariants are equal for all ids of a bin, a record listing
#! everything is added to the returned list. The invariants/characteristics are:
#! * the rank of the group
#! * the p-class of the group
#! * the information if the groups are isoclinic
#! * the size of the group of automorphisms
#! * the information if the groups are outer-automorphism-equivalent
#! * the information if the groups form a modular isomorphic pair (MIP)
#! The entries <A>isoclinic</A> and <A>MIP</A> are not deteremind in this function but could be added
#! to the entries of a table via `DetermineMIP` and `DetermineIsoclinic`.
DeclareGlobalFunction("TableOfGroupInvariants");

#! @Arguments tab
#! @Returns A list of records
#! @Description
#! Determines if the sets of groups represented by the records in <A>tab</A> are modular
#! isomorphic pairs using the &ModIsom; package. This information is added to each record
#! and the updated list is returned.
DeclareGlobalFunction("DetermineMIP");

#! @Arguments tab
#! @Returns A list of records
#! @Description
#! Determines if the sets of groups represented by the records in <A>tab</A> are
#! isoclinic using the &XMod; package. This information is added to each record and the
#! updated list is returned.
DeclareGlobalFunction("DetermineIsoclinic");

#! @Arguments tab1, tab2
#! @Returns A list of records
#! @Description
#! Merges the tables <A>tab1</A> and <A>tab2</A> into one table. If a record entry of
#! only one is empty the non-empty entry is used. This enables the parallel computation of
#! isoclinism and MIPs.
DeclareGlobalFunction("MergeTablesOfInvariants");

#! @Arguments tab, filename
#! @Description
#! Exports the content of the table <A>tab</A> into <A>filename</A> table.txt as
#! rows of a latex table.
DeclareGlobalFunction("PrintTableToFile");


